# TWSSH - High-Performance SSH Library for .NET

[![NuGet](https://img.shields.io/nuget/v/TwSsh.svg)](https://www.nuget.org/packages/TwSsh/)
[![License](https://img.shields.io/badge/license-Commercial-blue.svg)](LICENSE.txt)

TWSSH is a modern, high-performance SSH library for .NET applications. It provides comprehensive SSH2 protocol support including command execution, SFTP, SCP, port forwarding, and interactive shells.

## Features

- **SSH2 Protocol Support** - Full implementation of SSH2 protocol
- **Multiple Authentication Methods** - Password, public key, keyboard-interactive
- **SFTP Support** - Secure file transfer protocol
- **SCP Support** - Secure copy protocol
- **Port Forwarding** - Local and remote port forwarding
- **Interactive Shell** - Full PTY support for interactive sessions
- **Modern Cryptography** - ChaCha20-Poly1305, AES-GCM, Ed25519, ECDSA
- **Cross-Platform** - Works on Windows, Linux, and macOS
- **High Performance** - Optimized for low latency and high throughput
- **Async/Await** - Full async support with ValueTask

## Supported Frameworks

- .NET 8.0
- .NET 6.0
- .NET Standard 2.1

## Installation

```bash
dotnet add package TwSsh
```

## Quick Start

```csharp
using TwSsh.Client;
using TwSsh.Licensing;

// Initialize license (required - get a trial license at https://twssh.io)
LicenseManager.Instance.Initialize("YOUR-LICENSE-KEY");

// Connect and run a command
var settings = new SshClientSettings
{
    Host = "example.com",
    Port = 22,
    Credential = new PasswordCredential("username", "password")
};

await using var client = new SshClient(settings);
await client.ConnectAndAuthenticateAsync();

var result = await client.RunCommandAsync("uname -a");
Console.WriteLine(result.Output);
```

## Licensing & Pricing

TWSSH is commercial software. A valid license is required for production use.

### License Tiers

| License | Price | Updates | Connections | Features |
|---------|-------|---------|-------------|----------|
| **Trial** | Free | 30 days | 2 | All Team features |
| **Developer** | $99 | 2 years | 5 | SSH, Shell, SFTP, SCP, Public Key Auth |
| **Team** | $299 | 2 years | 25 | + Port Forwarding, KB-Interactive, Advanced Crypto |
| **Enterprise** | $749 | 2 years | Unlimited | + Priority Support, Connection Pooling |

All licenses are one-time purchases with 2 years of software updates included.

### Feature Comparison

| Feature | Developer | Team | Enterprise |
|---------|:---------:|:----:|:----------:|
| Basic SSH & Commands | ✓ | ✓ | ✓ |
| Interactive Shell | ✓ | ✓ | ✓ |
| SFTP File Transfer | ✓ | ✓ | ✓ |
| SCP File Transfer | ✓ | ✓ | ✓ |
| Public Key Auth | ✓ | ✓ | ✓ |
| Port Forwarding | | ✓ | ✓ |
| Keyboard-Interactive | | ✓ | ✓ |
| Advanced Crypto (ChaCha20) | | ✓ | ✓ |
| Connection Pooling | | ✓ | ✓ |
| Priority Support | | ✓ | ✓ |
| 4-Hour SLA Support | | | ✓ |
| Concurrent Connections | 5 | 25 | Unlimited |

### Obtaining a License

Contact us at **sales@twssh.io** to purchase a license.

### License Activation

```csharp
// Option 1: Initialize with license key
LicenseManager.Instance.Initialize("YOUR-LICENSE-KEY");

// Option 2: Use license file
LicenseManager.Instance.InitializeFromFile("path/to/license.lic");

// Option 3: Auto-discovery (place twssh.lic in app directory)
// The library automatically finds and loads it
```

Get a free 30-day trial license at [https://twssh.io](https://twssh.io)

## Documentation

Full documentation is available at [https://twssh.io/docs](https://twssh.io/docs)

## Support

- **Email**: support@twssh.io
- **Enterprise Support**: Priority response (4hr SLA) for Enterprise licenses

## Security

TWSSH implements modern security best practices:

- No deprecated algorithms (no MD5, SHA1, DES, 3DES)
- Support for latest key exchange algorithms
- Secure memory handling for credentials
- Regular security updates

## Changelog

### v1.2.8

- Fixed documentation inconsistencies and API accuracy
- Added comprehensive SCP and SFTP exception documentation
- Improved code examples for parallel command execution
- Updated feature availability matrix

### v1.2.7

- Fixed TwSsh.Contracts bundling in NuGet package
- Fixed sequential RunCommandAsync hanging issue
- Fixed channel disposal exception on close
- Updated documentation for API accuracy

### v1.2.6

- Performance improvements for large file transfers
- Enhanced connection stability under high load
- Fixed edge case in keyboard-interactive authentication
- Updated documentation and examples

### v1.2.0

- Added connection pooling for Enterprise licenses
- Improved SFTP resume capabilities
- Enhanced error messages and logging
- Performance optimizations for multi-channel scenarios

### v1.1.0

- Added keyboard-interactive authentication support
- ChaCha20-Poly1305 cipher implementation
- Improved PTY resize handling
- Bug fixes for remote port forwarding

### v1.0.0

- Initial release
- SSH2 protocol support
- Password and public key authentication
- Command execution and shell support
- SFTP and SCP file transfer
- Port forwarding (local and remote)
- Modern cryptographic algorithms
- Commercial licensing system

---

Copyright 2025 TWSSH. All rights reserved.
