# TWSSH Class Reference

Quick reference guide for all TWSSH public classes and types.

---

## Namespace: TwSsh.Client

### SshClient

The main class for SSH connections.

```csharp
public sealed class SshClient : IAsyncDisposable
```

| Member | Type | Description |
|--------|------|-------------|
| `State` | Property | Current connection state (`SshClientState`) |
| `IsConnected` | Property | Whether connected to server |
| `IsAuthenticated` | Property | Whether authenticated |
| `ConnectionInfo` | Property | Connection details (`SshConnectionInfo?`) |
| `Disconnected` | Event | Raised when disconnected |
| `HostKeyReceived` | Event | Raised for host key verification |
| `ConnectAsync()` | Method | Connect to SSH server |
| `ConnectAndAuthenticateAsync()` | Method | Connect and authenticate |
| `AuthenticateAsync()` | Method | Authenticate with credential |
| `DisconnectAsync()` | Method | Disconnect from server |
| `CreateCommandAsync()` | Method | Create command object |
| `RunCommandAsync()` | Method | Execute command and get result |
| `CreateShellAsync()` | Method | Create interactive shell |
| `OpenSubsystemChannelAsync()` | Method | Open subsystem (SFTP, etc.) |
| `ForwardLocalPortAsync()` | Method | Create local port forward |
| `ForwardRemotePortAsync()` | Method | Create remote port forward |

---

### SshClientSettings

Configuration for SSH connections.

```csharp
public sealed class SshClientSettings
```

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `Host` | `string` | `""` | Remote host |
| `Port` | `int` | `22` | Remote port |
| `Credential` | `SshCredential?` | `null` | Authentication credential |
| `ConnectionTimeout` | `TimeSpan` | 30s | Connection timeout |
| `OperationTimeout` | `TimeSpan` | 1min | Operation timeout |
| `KeepAliveInterval` | `TimeSpan` | Zero | Keep-alive interval |
| `AutoReconnect` | `bool` | `false` | Auto-reconnect |
| `ClientVersion` | `string?` | `null` | Custom version string |
| `HostKeyVerification` | `HostKeyVerificationMode` | `AutoAdd` | Host key mode |
| `HostKeyCallback` | `Func<...>?` | `null` | Custom verification |
| `Logger` | `ISshLogger?` | `null` | Custom logger |
| `KeyExchangeAlgorithms` | `IList<string>?` | `null` | Key exchange prefs |
| `HostKeyAlgorithms` | `IList<string>?` | `null` | Host key prefs |
| `EncryptionAlgorithms` | `IList<string>?` | `null` | Encryption prefs |
| `MacAlgorithms` | `IList<string>?` | `null` | MAC prefs |
| `CompressionAlgorithms` | `IList<string>?` | `null` | Compression prefs |

---

### SshConnectionInfo

Information about an established connection.

```csharp
public sealed class SshConnectionInfo
```

| Property | Type | Description |
|----------|------|-------------|
| `Host` | `string` | Remote host |
| `Port` | `int` | Remote port |
| `ServerVersion` | `string` | Server SSH version |
| `ClientVersion` | `string` | Client SSH version |
| `SessionId` | `byte[]` | Session identifier |
| `Algorithms` | `SshAlgorithmSet` | Negotiated algorithms |
| `Username` | `string` | Authenticated username |
| `IsAuthenticated` | `bool` | Whether authenticated |

---

### SshCommand

Represents a command execution.

```csharp
public sealed class SshCommand : IAsyncDisposable
```

| Member | Type | Description |
|--------|------|-------------|
| `CommandText` | Property | Command to execute |
| `OutputStream` | Property | Stdout stream |
| `ErrorStream` | Property | Stderr stream |
| `ExitStatus` | Property | Exit code (null if running) |
| `HasExecuted` | Property | Whether executed |
| `ExecuteAsync()` | Method | Execute and get result |
| `ExecuteAndGetOutputAsync()` | Method | Execute and get stdout |
| `GetOutputString()` | Method | Get current stdout |
| `GetErrorString()` | Method | Get current stderr |

---

### SshCommandResult

Result from command execution.

```csharp
public sealed class SshCommandResult
```

| Property | Type | Description |
|----------|------|-------------|
| `Output` | `string` | Stdout content |
| `Error` | `string` | Stderr content |
| `ErrorOutput` | `string` | Alias for Error |
| `ExitStatus` | `int` | Exit code |
| `ExitCode` | `int` | Alias for ExitStatus |
| `IsSuccess` | `bool` | True if ExitStatus is 0 |

---

### ShellStream

Interactive shell extending `System.IO.Stream`.

```csharp
public class ShellStream : Stream
```

| Member | Type | Description |
|--------|------|-------------|
| `Available` | Property | Bytes available to read |
| `DataAvailable` | Property | Whether data available |
| `Read()` | Method | Read bytes |
| `Write()` | Method | Write bytes |
| `ReadLine()` | Method | Read a line |
| `ReadLineAsync()` | Method | Read line async |
| `WriteLine()` | Method | Write text with newline |
| `WriteLineAsync()` | Method | Write line async |
| `Expect(string)` | Method | Wait for text |
| `Expect(Regex)` | Method | Wait for pattern |
| `ExpectAsync()` | Method | Wait async |
| `SendWindowChangeAsync()` | Method | Change terminal size |

---

### ForwardedPort

Base class for port forwarding.

```csharp
public abstract class ForwardedPort : IAsyncDisposable
```

| Member | Type | Description |
|--------|------|-------------|
| `IsStarted` | Property | Whether forwarding active |
| `ErrorOccurred` | Event | Raised on errors |
| `StopAsync()` | Method | Stop forwarding |

---

### LocalPortForwarder

Local port forwarding (SSH -L).

```csharp
public sealed class LocalPortForwarder : ForwardedPort
```

| Property | Type | Description |
|----------|------|-------------|
| `LocalHost` | `string` | Local bind host |
| `LocalPort` | `int` | Local bind port |
| `RemoteHost` | `string` | Remote destination |
| `RemotePort` | `int` | Remote port |

---

### RemotePortForwarder

Remote port forwarding (SSH -R).

```csharp
public sealed class RemotePortForwarder : ForwardedPort
```

| Property | Type | Description |
|----------|------|-------------|
| `RemoteHost` | `string` | Remote bind host |
| `RemotePort` | `int` | Remote bind port |
| `LocalHost` | `string` | Local destination |
| `LocalPort` | `int` | Local port |

---

## Namespace: TwSsh.Core.Authentication

### PrivateKeyParser

Static utility for parsing SSH private keys.

```csharp
public static class PrivateKeyParser
```

| Member | Type | Description |
|--------|------|-------------|
| `LoadFromFile()` | Static | Load key from file path |
| `Parse()` | Static | Parse key from string data |
| `ParseBinary()` | Static | Parse key from binary data |

Supported formats: OpenSSH, PEM (RSA, EC), PKCS#8, PKCS#8 Encrypted

---

## Namespace: TwSsh.Connection

### PtyRequest

PTY (pseudo-terminal) configuration for shells.

```csharp
public sealed class PtyRequest
```

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `TerminalType` | `string` | `"xterm"` | Terminal type |
| `WidthChars` | `uint` | `80` | Width in characters |
| `Columns` | `uint` | `80` | Alias for WidthChars |
| `HeightRows` | `uint` | `24` | Height in rows |
| `Rows` | `uint` | `24` | Alias for HeightRows |
| `WidthPixels` | `uint` | `0` | Width in pixels |
| `HeightPixels` | `uint` | `0` | Height in pixels |
| `Modes` | `Dictionary<TerminalMode, uint>` | `new()` | Terminal modes |

**Note:** `Columns`/`Rows` and `WidthChars`/`HeightRows` are aliases - either works.

---

## Namespace: TwSsh.Authentication

### SshCredential

Base class for credentials.

```csharp
public abstract class SshCredential
```

| Property | Type | Description |
|----------|------|-------------|
| `Username` | `string` | Username |

---

### PasswordCredential

Password authentication.

```csharp
public sealed class PasswordCredential : SshCredential
```

| Member | Type | Description |
|--------|------|-------------|
| Constructor | | `new PasswordCredential(username, password)` |
| `GetPassword()` | Method | Get password string |

---

### PrivateKeyCredential

Private key authentication.

```csharp
public sealed class PrivateKeyCredential : SshCredential, IDisposable
```

| Member | Type | Description |
|--------|------|-------------|
| Constructor | | `new PrivateKeyCredential(username, keys...)` |
| `PrivateKeys` | Property | List of private keys |
| `Dispose()` | Method | Clear sensitive data |

---

### KeyboardInteractiveCredential

Keyboard-interactive authentication.

```csharp
public sealed class KeyboardInteractiveCredential : SshCredential
```

| Member | Type | Description |
|--------|------|-------------|
| Constructor | | `new KeyboardInteractiveCredential(username, handler)` |
| `PromptHandler` | Property | Handler for prompts |
| `FromPassword()` | Static | Create from password |

---

### SshPrivateKey

Abstract private key representation.

```csharp
public abstract class SshPrivateKey : IDisposable
```

| Member | Type | Description |
|--------|------|-------------|
| `Algorithm` | Property | Key algorithm name |
| `GetPublicKeyBlob()` | Method | Get public key |
| `Sign()` | Method | Sign data |
| `Dispose()` | Method | Clear sensitive data |

---

## Namespace: TwSsh.Sftp

### SftpClient

High-level SFTP client for file transfer operations.

```csharp
public sealed class SftpClient : IAsyncDisposable
```

| Member | Type | Description |
|--------|------|-------------|
| `ConnectAsync()` | Static | Create SFTP client from SSH connection |
| `IsConnected` | Property | Whether SFTP session active |
| `ProtocolVersion` | Property | Negotiated protocol version (3-6) |
| `Capabilities` | Property | Server capability detection |
| `UploadFileAsync()` | Method | Upload file to remote |
| `DownloadFileAsync()` | Method | Download file from remote |
| `UploadFilesAsync()` | Method | Batch upload files |
| `DownloadFilesAsync()` | Method | Batch download files |
| `SyncToRemoteAsync()` | Method | Sync directory to remote |
| `OpenFileAsync()` | Method | Open remote file as stream |
| `OpenReadAsync()` | Method | Open file for reading |
| `OpenWriteAsync()` | Method | Open file for writing |
| `ReadAllBytesAsync()` | Method | Read file as byte array |
| `WriteAllBytesAsync()` | Method | Write byte array to file |
| `DownloadToStreamAsync()` | Method | Download to stream |
| `UploadFromStreamAsync()` | Method | Upload from stream |
| `DeleteFileAsync()` | Method | Delete remote file |
| `RenameAsync()` | Method | Rename/move remote file |
| `PosixRenameAsync()` | Method | Atomic rename (requires extension) |
| `GetFileInfoAsync()` | Method | Get file information |
| `GetAttributesAsync()` | Method | Get file attributes |
| `SetAttributesAsync()` | Method | Set file attributes |
| `ExistsAsync()` | Method | Check if file exists |
| `GetRealPathAsync()` | Method | Get canonical path |
| `GetWorkingDirectoryAsync()` | Method | Get current directory |
| `CreateDirectoryAsync()` | Method | Create remote directory |
| `DeleteDirectoryAsync()` | Method | Delete remote directory |
| `ListDirectoryAsync()` | Method | List directory contents |
| `ListDirectoryInfoAsync()` | Method | List as SftpFileInfo |
| `EnumerateDirectoryAsync()` | Method | Stream directory entries |
| `CreateSymbolicLinkAsync()` | Method | Create symbolic link |
| `CreateHardLinkAsync()` | Method | Create hard link (requires extension) |
| `ReadSymbolicLinkAsync()` | Method | Read link target |
| `GetFilesystemInfoAsync()` | Method | Get filesystem statistics |

---

### SftpFileStream

Seekable stream for remote files.

```csharp
public sealed class SftpFileStream : Stream
```

| Member | Type | Description |
|--------|------|-------------|
| `CanRead` | Property | Whether stream can read |
| `CanWrite` | Property | Whether stream can write |
| `CanSeek` | Property | Always `true` |
| `Length` | Property | File length |
| `Position` | Property | Current position |
| Standard Stream methods | | Read, Write, Seek, SetLength, Flush |

---

### SftpFileInfo

Information about a remote file or directory.

```csharp
public sealed class SftpFileInfo
```

| Property | Type | Description |
|----------|------|-------------|
| `FullName` | `string` | Full path to file |
| `Name` | `string` | File name only |
| `Attributes` | `SftpFileAttributes` | File attributes |
| `IsDirectory` | `bool` | Whether directory |
| `IsFile` | `bool` | Whether regular file |
| `IsRegularFile` | `bool` | Alias for IsFile |
| `IsSymbolicLink` | `bool` | Whether symbolic link |
| `Length` | `long` | Size in bytes |
| `Size` | `long` | Alias for Length (both are equivalent) |
| `LastAccessTime` | `DateTime?` | Last access time |
| `LastWriteTime` | `DateTime?` | Last modification time |
| `Permissions` | `uint?` | Unix permissions |

---

### SftpDirectoryEntry

A single entry from directory listing.

```csharp
public sealed class SftpDirectoryEntry
```

| Property | Type | Description |
|----------|------|-------------|
| `FileName` | `string` | File or directory name |
| `LongName` | `string` | Long format (like `ls -l`) |
| `Attributes` | `SftpFileAttributes` | File attributes |
| `IsDirectory` | `bool` | Whether directory |
| `IsRegularFile` | `bool` | Whether regular file |
| `IsSymbolicLink` | `bool` | Whether symbolic link |
| `Size` | `long` | Size in bytes |
| `ModifyTime` | `DateTimeOffset?` | Last modification time |

---

### SftpTransferOptions

Options for file transfer operations.

```csharp
public sealed class SftpTransferOptions
```

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `BufferSize` | `int` | `32768` | Transfer buffer size |
| `Resume` | `bool` | `false` | Resume interrupted transfers |
| `StartOffset` | `long` | `0` | Offset to start from |
| `MaxBytes` | `long?` | `null` | Max bytes to transfer |
| `VerifySize` | `bool` | `true` | Verify file sizes |
| `Overwrite` | `bool` | `true` | Overwrite existing files |
| `PreserveTimestamps` | `bool` | `false` | Preserve timestamps |
| `PreservePermissions` | `bool` | `false` | Preserve permissions |
| `ParallelRequests` | `int` | `1` | Parallel requests |
| `ProgressInterval` | `long` | `0` | Progress report interval |
| `Default` | Static | | Default options |
| `ResumeDownload()` | Static | | Create resume download options |
| `ResumeUpload()` | Static | | Create resume upload options |
| `Parallel()` | Static | | Create parallel transfer options |

---

### SftpTransferProgress

Progress information for transfers.

```csharp
public sealed class SftpTransferProgress
```

| Property | Type | Description |
|----------|------|-------------|
| `FileName` | `string` | File being transferred |
| `BytesTransferred` | `long` | Bytes transferred |
| `TotalBytes` | `long` | Total bytes |
| `Elapsed` | `TimeSpan` | Time elapsed |
| `PercentComplete` | `double` | Completion % (0-100) |
| `BytesPerSecond` | `double` | Transfer speed |
| `EstimatedRemaining` | `TimeSpan?` | ETA (null if unknown) |
| `IsComplete` | `bool` | Whether complete |

---

### SftpCapabilities

Server capability detection.

```csharp
public sealed class SftpCapabilities
```

| Property | Type | Description |
|----------|------|-------------|
| `ProtocolVersion` | `int` | Negotiated version |
| `SupportsV3` | `bool` | Version 3 support |
| `SupportsV4` | `bool` | Version 4 support |
| `SupportsV5` | `bool` | Version 5 support |
| `SupportsV6` | `bool` | Version 6 support |
| `SupportsPosixRename` | `bool` | posix-rename@openssh.com extension |
| `SupportsStatVfs` | `bool` | statvfs@openssh.com extension |
| `SupportsFStatVfs` | `bool` | fstatvfs@openssh.com extension |
| `SupportsHardLink` | `bool` | hardlink@openssh.com extension |
| `SupportsFsync` | `bool` | fsync@openssh.com extension |
| `SupportsSymlinks` | `bool` | Symlink support (v3+) |
| `SupportsFileType` | `bool` | File type in attributes (v4+) |
| `SupportsOwnerGroupNames` | `bool` | Owner/group names (v4+) |
| `SupportsCreationTime` | `bool` | Creation time (v4+) |
| `SupportsRenameFlags` | `bool` | Atomic rename (v5+) |
| `SupportsTextMode` | `bool` | Text mode transfers (v5+) |
| `SupportsFileLocking` | `bool` | File locking (v6+) |
| `SupportsLinkOperation` | `bool` | Hard links in protocol (v6+) |
| `SupportsExtension()` | Method | Check any extension by name |
| `GetExtensionVersion()` | Method | Get extension version string |
| `GetSupportedExtensions()` | Method | Get all extensions dictionary |

---

### SftpStatVfs

Filesystem statistics.

```csharp
public sealed class SftpStatVfs
```

| Property | Type | Description |
|----------|------|-------------|
| `BlockSize` | `ulong` | Block size |
| `FragmentSize` | `ulong` | Fragment size |
| `TotalBlocks` | `ulong` | Total blocks |
| `FreeBlocks` | `ulong` | Free blocks |
| `AvailableBlocks` | `ulong` | Available blocks |
| `TotalSize` | `ulong` | Total size in bytes |
| `FreeSize` | `ulong` | Free space in bytes |
| `AvailableSize` | `ulong` | Available space in bytes |
| `UsedSize` | `ulong` | Used space in bytes |
| `UsagePercent` | `double` | Usage percentage (0-100) |
| `TotalInodes` | `ulong` | Total inodes |
| `FreeInodes` | `ulong` | Free inodes |
| `AvailableInodes` | `ulong` | Available inodes |
| `FileSystemId` | `ulong` | Filesystem ID |
| `MountFlags` | `ulong` | Mount flags |
| `MaxNameLength` | `ulong` | Max filename length |

---

## Namespace: TwSsh.Scp

### ScpClient

SCP client for simple file copy operations.

```csharp
public sealed class ScpClient : IAsyncDisposable
```

| Member | Type | Description |
|--------|------|-------------|
| `ConnectAsync()` | Static | Create SCP client from SSH connection |
| `IsConnected` | Property | Whether SCP session active |
| `UploadAsync()` | Method | Upload file to remote |
| `DownloadAsync()` | Method | Download file from remote |

---

### ScpTransferProgress

Progress information for SCP transfers.

```csharp
public sealed class ScpTransferProgress
```

| Property | Type | Description |
|----------|------|-------------|
| `FileName` | `string` | File being transferred |
| `BytesTransferred` | `long` | Bytes transferred |
| `TotalBytes` | `long` | Total bytes |
| `PercentComplete` | `double` | Completion % (0-100) |

---

## Namespace: TwSsh.Licensing

### LicenseManager

Singleton license manager.

```csharp
public sealed class LicenseManager
```

| Member | Type | Description |
|--------|------|-------------|
| `Instance` | Static | Singleton instance |
| `IsInitialized` | Property | Whether initialized |
| `LicenseInfo` | Property | License details (`LicenseInfo?`) |
| `CurrentLicense` | Property | Alias for LicenseInfo |
| `Initialize()` | Method | Initialize with key string |
| `InitializeFromFile()` | Method | Load from file |
| `TryAutoInitialize()` | Method | Auto-discover and load license |
| `HasFeature()` | Method | Check feature availability |
| `RequireFeature()` | Method | Require feature or throw |

---

### LicenseInfo

License information returned by `LicenseManager.LicenseInfo`.

```csharp
public sealed class LicenseInfo
```

| Property | Type | Description |
|----------|------|-------------|
| `Type` | `LicenseType` | License type (Trial, Developer, Team, Enterprise) |
| `LicenseeName` | `string` | Name of the licensee |
| `LicenseeEmail` | `string?` | Email of the licensee |
| `ExpirationDate` | `DateTime?` | License expiration date (null = perpetual) |
| `Expires` | `DateTime?` | Alias for ExpirationDate |
| `IsValid` | `bool` | Whether the license is currently valid |
| `IsExpired` | `bool` | Whether the license has expired |
| `DaysRemaining` | `int?` | Days until expiration (null if perpetual) |
| `MaxConnections` | `int` | Maximum concurrent connections allowed |
| `Features` | `LicenseFeatures` | Licensed features flags |
| `IssueDate` | `DateTime` | When the license was issued |

#### Example

```csharp
var license = LicenseManager.Instance.LicenseInfo;

if (license != null)
{
    Console.WriteLine($"Type: {license.Type}");
    Console.WriteLine($"Licensed to: {license.LicenseeName}");
    Console.WriteLine($"Expires: {license.ExpirationDate?.ToString("yyyy-MM-dd") ?? "Never"}");
    Console.WriteLine($"Days remaining: {license.DaysRemaining}");
    Console.WriteLine($"Max connections: {license.MaxConnections}");
    Console.WriteLine($"Valid: {license.IsValid}");
}
```

---

## Enumerations

### SshClientState

```csharp
public enum SshClientState
{
    Disconnected,   // Not connected
    Connecting,     // Connecting
    Connected,      // Connected, not authenticated
    Authenticated,  // Ready for operations
    Disconnecting   // Disconnecting
}
```

### HostKeyVerificationMode

```csharp
public enum HostKeyVerificationMode
{
    None,      // Accept all (insecure!)
    AutoAdd,   // Trust on first use
    Strict,    // Reject unknown
    Callback   // Use custom callback
}
```

### LicenseFeatures

```csharp
[Flags]
public enum LicenseFeatures
{
    None = 0,
    BasicSsh = 1,          // Basic SSH connection and command execution
    Shell = 2,             // Interactive shell (ShellStream)
    Sftp = 4,              // Full SftpClient functionality
    Scp = 8,               // SCP operations
    PublicKeyAuth = 16,    // Public key authentication
    PortForwarding = 32,   // Local and remote port forwarding
    KeyboardInteractive = 64,  // Keyboard-interactive authentication
    AdvancedCrypto = 128,  // ChaCha20-Poly1305, advanced algorithms
    ConnectionPooling = 256,   // Connection pooling for high-throughput scenarios
    PrioritySupport = 512  // Priority support entitlement
}
```

**Feature Availability by License Tier:**

| Feature | Developer | Team | Enterprise | Trial |
|---------|:---------:|:----:|:----------:|:-----:|
| BasicSsh | ✓ | ✓ | ✓ | ✓ |
| Shell | ✓ | ✓ | ✓ | ✓ |
| Sftp | ✓ | ✓ | ✓ | ✓ |
| Scp | ✓ | ✓ | ✓ | ✓ |
| PublicKeyAuth | ✓ | ✓ | ✓ | ✓ |
| PortForwarding | | ✓ | ✓ | ✓ |
| KeyboardInteractive | | ✓ | ✓ | ✓ |
| AdvancedCrypto | | ✓ | ✓ | ✓ |
| ConnectionPooling | | ✓ | ✓ | ✓ |
| PrioritySupport | | ✓ | ✓ | |

**Note:** Trial licenses include all Team-tier features for full evaluation purposes. The SFTP feature enables full `SftpClient` functionality. See [Namespace: TwSsh.Sftp](#namespace-twsshsftp).

---

### SshDisconnectReason

SSH disconnect reason codes (RFC 4253).

```csharp
public enum SshDisconnectReason : uint
{
    None = 0,
    HostNotAllowedToConnect = 1,
    ProtocolError = 2,
    KeyExchangeFailed = 3,
    Reserved = 4,
    MacError = 5,
    CompressionError = 6,
    ServiceNotAvailable = 7,
    ProtocolVersionNotSupported = 8,
    HostKeyNotVerifiable = 9,
    ConnectionLost = 10,
    ByApplication = 11,
    TooManyConnections = 12,
    AuthCancelledByUser = 13,
    NoMoreAuthMethodsAvailable = 14,
    IllegalUserName = 15
}
```

**Common Disconnect Reasons:**

| Value | Description |
|-------|-------------|
| `ByApplication` | Normal disconnect initiated by client or server |
| `AuthCancelledByUser` | User cancelled authentication |
| `NoMoreAuthMethodsAvailable` | All authentication methods exhausted |
| `ConnectionLost` | Network connection was lost |
| `ProtocolError` | SSH protocol violation detected |
| `HostKeyNotVerifiable` | Host key verification failed |

---

## Namespace: TwSsh.Common

### SshException

Base SSH exception.

```csharp
public class SshException : Exception
```

### SshConnectionException

Connection failures.

```csharp
public class SshConnectionException : SshException
```

### SshAuthenticationException

Authentication failures.

```csharp
public class SshAuthenticationException : SshException
```

| Property | Type | Description |
|----------|------|-------------|
| `AllowedMethods` | `IReadOnlyList<string>?` | Authentication methods allowed by server |

The `AllowedMethods` property contains the list of authentication methods the server accepts. Use this to determine fallback options:

```csharp
try
{
    await client.ConnectAndAuthenticateAsync();
}
catch (SshAuthenticationException ex)
{
    Console.WriteLine($"Authentication failed: {ex.Message}");
    if (ex.AllowedMethods != null)
    {
        Console.WriteLine($"Server accepts: {string.Join(", ", ex.AllowedMethods)}");
    }
}
```

### SshChannelException

Channel operation failures.

```csharp
public class SshChannelException : SshException
```

| Property | Type | Description |
|----------|------|-------------|
| `ChannelId` | `uint?` | Channel ID that failed |
| `ReasonCode` | `SshChannelOpenFailureReason?` | Failure reason |

### SftpException

SFTP-specific errors with status codes.

```csharp
public class SftpException : SshException
```

| Property | Type | Description |
|----------|------|-------------|
| `StatusCode` | `SftpStatusCode` | SFTP status code |

**Note:** The error message typically contains path information when relevant.

### SftpStatusCode

SFTP protocol status codes.

```csharp
public enum SftpStatusCode
{
    Ok = 0,
    Eof = 1,
    NoSuchFile = 2,
    PermissionDenied = 3,
    Failure = 4,
    // ... see API-REFERENCE.md for full list
}
```

### SshChannelOpenFailureReason

Reason codes for channel open failures.

```csharp
public enum SshChannelOpenFailureReason : uint
{
    AdministrativelyProhibited = 1,
    ConnectFailed = 2,
    UnknownChannelType = 3,
    ResourceShortage = 4
}
```

---

## Event Arguments

### SshDisconnectEventArgs

```csharp
public class SshDisconnectEventArgs : EventArgs
{
    public SshDisconnectReason Reason { get; }
}
```

**Note:** The `Reason` property contains the SSH disconnect reason code. Use the enum value's name or description for display purposes.

### HostKeyEventArgs

```csharp
public class HostKeyEventArgs : EventArgs
{
    public string Host { get; }
    public string KeyType { get; }
    public byte[] PublicKey { get; }
    public string Fingerprint { get; }
    public bool Accept { get; set; }
}
```

### ChannelDataEventArgs

```csharp
public class ChannelDataEventArgs : EventArgs
{
    public ReadOnlyMemory<byte> Data { get; }
}
```

---

## Interfaces

### ISessionChannel

Session channel for commands, shell, and subsystems.

```csharp
public interface ISessionChannel : ISshChannel
{
    // Request operations
    ValueTask<bool> RequestExecAsync(string command, CancellationToken ct);
    ValueTask<bool> RequestShellAsync(CancellationToken ct);
    ValueTask<bool> RequestSubsystemAsync(string name, CancellationToken ct);
    ValueTask<bool> RequestPtyAsync(PtyRequest request, CancellationToken ct);

    // Environment and signals
    ValueTask<bool> SetEnvironmentVariableAsync(string name, string value, CancellationToken ct);
    ValueTask SendSignalAsync(string signalName, CancellationToken ct);

    // Window size change (full control)
    ValueTask SendWindowChangeAsync(uint widthChars, uint heightRows, uint widthPixels = 0, uint heightPixels = 0, CancellationToken ct = default);

    // Exit status
    int? ExitStatus { get; }
    string? ExitSignal { get; }

    // Events
    event EventHandler<ChannelDataEventArgs> DataReceived;
    event EventHandler<ExitStatusEventArgs> ExitStatusReceived;
    event EventHandler<ExitSignalEventArgs> ExitSignalReceived;
}
```

### ISshChannel

Base channel interface.

```csharp
public interface ISshChannel : IAsyncDisposable
{
    uint LocalChannelId { get; }
    uint RemoteChannelId { get; }
    SshChannelState State { get; }
    ValueTask WriteAsync(ReadOnlyMemory<byte> data, CancellationToken ct);
    ValueTask<int> ReadAsync(Memory<byte> buffer, CancellationToken ct);
    ValueTask CloseAsync(CancellationToken ct);
    ValueTask SendEofAsync(CancellationToken ct);
}
```

---

## Type Aliases

For convenience, these types are commonly used:

| Type | Full Name |
|------|-----------|
| `SshClient` | `TwSsh.Client.SshClient` |
| `SshClientSettings` | `TwSsh.Client.SshClientSettings` |
| `ShellStream` | `TwSsh.Client.ShellStream` |
| `PasswordCredential` | `TwSsh.Authentication.PasswordCredential` |
| `PrivateKeyCredential` | `TwSsh.Authentication.PrivateKeyCredential` |
| `PrivateKeyParser` | `TwSsh.Core.Authentication.PrivateKeyParser` |
| `PtyRequest` | `TwSsh.Connection.PtyRequest` |
| `SftpClient` | `TwSsh.Sftp.SftpClient` |
| `SftpFileStream` | `TwSsh.Sftp.SftpFileStream` |
| `SftpFileInfo` | `TwSsh.Sftp.SftpFileInfo` |
| `SftpTransferOptions` | `TwSsh.Sftp.SftpTransferOptions` |
| `SftpException` | `TwSsh.Sftp.SftpException` |
| `ScpClient` | `TwSsh.Scp.ScpClient` |
| `ScpTransferProgress` | `TwSsh.Scp.ScpTransferProgress` |
| `LicenseManager` | `TwSsh.Licensing.LicenseManager` |

---

## Common Using Statements

```csharp
using TwSsh.Client;
using TwSsh.Authentication;
using TwSsh.Core.Authentication;  // For PrivateKeyParser
using TwSsh.Connection;           // For PtyRequest
using TwSsh.Sftp;                 // For SftpClient, SftpException
using TwSsh.Scp;                  // For ScpClient
using TwSsh.Licensing;
using TwSsh.Common;               // For SshException, ISshLogger
```

---

For detailed documentation, see [API-REFERENCE.md](API-REFERENCE.md).

---

Copyright 2025 Terminalworks Ltd. All rights reserved.
